import { PrismaClient } from '@prisma/pg/client';
import fs from 'fs';
const prisma = new PrismaClient();

function getCurrentUtcTime() {
  return new Date().toISOString();
}

const configurationsSeed = JSON.parse(
  fs.readFileSync(__dirname + '/json/configurations.json', 'utf-8'),
);

/*let catIdGenerator = {
  prefix: 'a',
  currentIndex: 1,
};

// Function to generate a new cat_id
function generateCatId() {
  const catId = `${catIdGenerator.prefix}${catIdGenerator.currentIndex}`;
  catIdGenerator.currentIndex++; // Increment the index for the next id
  return catId;
}*/

export async function configurationsSeeder(company_id: number) {
  try {
    let user = await prisma.user.findFirst({
      where: { company_id: company_id, role: 'super_admin' },
    });
    if (!user) {
      console.log(
        `Role seeder failed: admin user not available for company id ${company_id}`,
      );
      return false;
    }

    const createdBy = user.id;
    const utcNow = getCurrentUtcTime();

    await prisma.$transaction(async (tx) => {
      for (const catType of configurationsSeed) {
        // 1. CATEGORY TYPE
        const existingCategoryType = await tx.categoryTypes.findUnique({
          where: {
            company_id_name: { company_id, name: catType.cat_type_name },
          },
        });
        if (existingCategoryType) {
          // Delete all categories and configurations under this type first
          const categoriesToDelete = await tx.categories.findMany({
            where: {
              company_id,
              category_type_id: existingCategoryType.category_type_id,
            },
            select: { category_id: true },
          });
          for (const cat of categoriesToDelete) {
            await tx.configurations.deleteMany({
              where: { company_id, category_id: cat.category_id },
            });
          }
          await tx.categories.deleteMany({
            where: {
              company_id,
              category_type_id: existingCategoryType.category_type_id,
            },
          });
          await tx.categoryTypes.delete({
            where: {
              company_id_name: { company_id, name: catType.cat_type_name },
            },
          });
        }
        // Insert new CategoryType
        const categoryType = await tx.categoryTypes.create({
          data: {
            category_type_id: catType.category_type_id,
            name: catType.cat_type_name,
            description: catType.cat_type_description,
            company_id: company_id,
            created_by: createdBy,
            created_at: utcNow,
          },
        });

        if (catType.cats && catType.cats.length) {
          for (const category of catType.cats) {
            // 2. CATEGORY
            const existingCategory = await tx.categories.findUnique({
              where: {
                company_id_name: { company_id, name: category.cat_name },
              },
            });
            if (existingCategory) {
              // Delete all configurations under this category
              await tx.configurations.deleteMany({
                where: { company_id, category_id: existingCategory.category_id },
              });
              await tx.categories.delete({
                where: {
                  company_id_name: { company_id, name: category.cat_name },
                },
              });
            }
            // Insert new Category
            const newCategory = await tx.categories.create({
              data: {
                category_id: category.category_id,
                name: category.cat_name,
                description: category.cat_description,
                category_type_id: categoryType.category_type_id,
                company_id: company_id,
                created_by: createdBy,
                created_at: utcNow,
              },
            });

            // 3. CONFIGURATIONS
            if (category.cat_configs && category.cat_configs.length) {
              for (const config of category.cat_configs) {
                const existingConfig = await tx.configurations.findUnique({
                  where: {
                    company_id_config_key: {
                      company_id,
                      config_key: config.config_key,
                    },
                  },
                });
                if (existingConfig) {
                  await tx.configurations.delete({
                    where: {
                      company_id_config_key: {
                        company_id,
                        config_key: config.config_key,
                      },
                    },
                  });
                }
                const configFileValue: any = config.config_file_value;
                await tx.configurations.create({
                  data: {
                    order_no: config.order_no,
                    config_key: config.config_key,
                    config_value: config.config_value
                      ? config.config_value
                      : 'null',
                    config_file_value: configFileValue,
                    config_value_type: config.config_value_type,
                    config_field_type: config.config_field_type,
                    category_id: newCategory.category_id,
                    company_id: company_id,
                    created_by: createdBy,
                    created_at: utcNow,
                  },
                });
              }
            }
          }
        }
      }
    });
  } catch (error) {
    console.error(`Error in configurationsSeeder: ${error}`);
    throw error;
  }
}
