import { PrismaClient } from '@prisma/pg/client';
import fs from 'fs';
const prisma = new PrismaClient();

export async function appImportTemplatesSeeder(company_id: number) {
  try {
    const importTemplates = JSON.parse(
      fs.readFileSync(__dirname + '/json/import-templates.json', 'utf-8'),
    );

    for (const importTemplate of importTemplates) {
      const { lineitems, queries, ...importTemplateData } = importTemplate;

      let user = await prisma.user.findFirst({
        where: { company_id, role: 'super_admin' },
      });
      if (!user) {
        console.log(
          `Import template seeder failed: admin user not available for company id ${company_id}`,
        );
        return false;
      }

      // Upsert Import Template
      const mg = await prisma.importTemplate.upsert({
        where: {
          company_id_slug: {
            company_id,
            slug: importTemplateData.slug,
          },
        },
        update: {
          ...importTemplateData,
          updated_by: user.id,
          updated_at: new Date(),
        },
        create: {
          ...importTemplateData,
          company_id,
          created_by: user.id,
        },
      });

      // Upsert LineItems
      for (const lineitem of lineitems) {
        let field_type = await prisma.fieldType.findFirst({
          where: {
            company_id,
            field_type: {
              equals: lineitem.field_type_id,
              mode: 'insensitive', // Case-insensitive comparison
            },
          },
        });
        if (!field_type) {
          console.log(
            `Import template seeder failed: ${lineitem.field_type_id} field_type not available for company id ${company_id}`,
          );
          return false;
        }

        await prisma.importTemplateLineItem.upsert({
          where: {
            company_id_import_template_id_display_name: {
              company_id,
              import_template_id: mg.id,
              display_name: lineitem.display_name,
            },
          },
          update: {
            ...lineitem,
            import_template_id: mg.id,
            company_id,
            field_type_id: field_type.id,
          },
          create: {
            ...lineitem,
            import_template_id: mg.id,
            company_id,
            field_type_id: field_type.id,
          },
        });
      }

      // Upsert Queries (New Addition)
      for (const query of queries) {
        // Check if the query already exists
        const existingQuery = await prisma.importTemplateQueries.findFirst({
          where: {
            company_id,
            import_template_id: mg.id,
            query_name: query.query_name,
          },
        });

        if (existingQuery) {
          // Update if it exists
          await prisma.importTemplateQueries.update({
            where: { id: existingQuery.id }, // Use the unique identifier to update
            data: {
              ...query,
              import_template_id: mg.id,
              company_id,
            },
          });
        } else {
          // Insert if it does not exist
          await prisma.importTemplateQueries.create({
            data: {
              ...query,
              import_template_id: mg.id,
              company_id,
            },
          });
        }
      }
    }

    console.log('Import template creation completed.');
  } catch (error) {
    console.log(`Import template seeder failed.`);
    throw error;
  }
}
