import { PrismaClient, Prisma } from '@prisma/pg/client';
import fs from 'fs';
import path from 'path';

const prisma = new PrismaClient();

function formatDisplayName(displayName: string): string {
  // Replace underscores with spaces and capitalize each word
  return displayName
    .replace(/_/g, ' ') // Replace underscores with spaces
    .toLowerCase() // Convert to lower case
    .replace(/\b\w/g, (char) => char.toUpperCase()); // Capitalize the first letter of each word
}

async function languageContent(
  languageContentData: { [key: string]: string },
  companyId: number,
  languageId: number,
) {
  try {
    let user = await prisma.user.findFirst({
      where: { company_id: companyId, role: 'super_admin' },
    });
    if (!user) {
      console.log(
        `Menu seeder failed: admin user not available for company id ${companyId}`,
      );
      return false;
    }

    const createdBy = user.id;

    const company = await prisma.user.findFirst({
      where: { company_id: companyId },
    });

    if (!company) {
      console.log(
        `WARNING: languageSeeds: Company with id ${companyId} does not exist.`,
      );
      return;
    }

    const distinctEntityNames = await prisma.masterEntityLineItem.findMany({
      where: {
        company_id: companyId,
        status_id: 1,
      },
      distinct: ['display_name'],
      select: {
        display_name: true,
      },
    });

    const entityNames = distinctEntityNames.map((item) => item.display_name);

    const menuNames = await prisma.menus.findMany({
      where: {
        company_id: companyId,
        status_id: 1,
      },
      select: {
        name: true,
      },
    });

    const menuNameList = menuNames.map((item) => item.name);

    const menuItemNames = await prisma.menuItems.findMany({
      where: {
        company_id: companyId,
        status_id: 1,
      },
      select: {
        name: true,
      },
      orderBy: {
        order_no: 'asc',
      },
    });

    const menuItemNameList = menuItemNames.map((item) => item.name);

    const displayNames = [...entityNames, ...menuNameList, ...menuItemNameList];

    const keyValueMapping: { [key: string]: string } = {};

    /*for (const displayName of displayNames) {
      const formattedValue = formatDisplayName(displayName);
      keyValueMapping[displayName] = formattedValue;
    }*/

    for (const [key, value] of Object.entries(languageContentData)) {
      keyValueMapping[key] = value as string;
    }

    for (const [key, value] of Object.entries(keyValueMapping)) {
      // Check if the key already exists for this language and company
      const existingContent = await prisma.languageContent.findFirst({
        where: {
          language_id: languageId,
          company_id: companyId,
          key_content: key,
        },
      });

      // Only create if the key doesn't exist
      if (!existingContent) {
        await prisma.languageContent.create({
          data: {
            language_id: languageId,
            key_content: key,
            values: value,
            company_id: companyId,
          },
        });
        console.log(
          `Created new languageContent: key ${key}, value ${value}, company_id ${companyId}`,
        );
      } else {
        console.log(
          `Skipped existing languageContent: key ${key}, company_id ${companyId}`,
        );
      }
    }
  } catch (error) {
    console.error(`Error in languageContentSeeds: ${error}`);
    if (
      error instanceof Prisma.PrismaClientKnownRequestError &&
      error.code === 'P2003' &&
      error.meta
    ) {
      console.error(`Foreign key constraint failed: ${error.meta.field_name}`);
    }
    throw error;
  } finally {
    // await prisma.$disconnect();
  }
}

export async function appLanguageContentSeeds(company_id: number) {
  try {
    const filePathEn = path.join(
      __dirname,
      '/json/language-content-english.json',
    );
    const languageContentListEn: { [key: string]: string }[] = JSON.parse(
      fs.readFileSync(filePathEn, 'utf-8'),
    );

    for (const languageContentDataEn of languageContentListEn) {
      await languageContent(languageContentDataEn, company_id, 1);
    }

    const filePathAr = path.join(
      __dirname,
      '/json/language-content-arabic.json',
    );
    const languageContentListAr: { [key: string]: string }[] = JSON.parse(
      fs.readFileSync(filePathAr, 'utf-8'),
    );

    for (const languageContentDataAr of languageContentListAr) {
      await languageContent(languageContentDataAr, company_id, 2);
    }

    console.log('language Content seeds completed.');
  } catch (error) {
    throw error;
  }
}
