import { PrismaClient } from '@prisma/sql/client';
import fs from 'fs';
import path from 'path';

const prisma = new PrismaClient();

async function getMasterEntityIdBySlug(permission_slug: string, company_id: number) {
  const masterEntity = await prisma.masterEntity.findFirst({
    where: {
      company_id,
      entity_name: permission_slug,
    },
  });
  return masterEntity ? masterEntity.id : null;
}

async function getMasterPermissionIdBySlug(permission_slug: string, company_id: number) {
  const masterPermission = await prisma.permissions.findFirst({
    where: {
      company_id,
      slug: permission_slug,
    },
  });
  return masterPermission ? masterPermission.id : null;
}

async function createMenu(company_id: number, menu: any, created_by: number) {
  const { children, entity_name, permission_slug, ...menuData } = menu;

  let existingMenu = await prisma.menuItems.findFirst({
    where: { company_id, name: menu.name, parent_id: menuData.parent_id, menu_id: menuData.menu_id },
  });

  let entity_id: number | null = null;
  if (entity_name) {
    entity_id = await getMasterEntityIdBySlug(entity_name, company_id);
  }

  let permission_id: number | null = null;
  if (permission_slug) {
    permission_id = await getMasterPermissionIdBySlug(permission_slug, company_id);
  }

  if (!existingMenu) {
    existingMenu = await prisma.menuItems.create({
      data: {
        menu_id: menu.menu_id,
        name: menu.name,
        target: menu.target,
        permission_id: permission_id,
        menu_img: menu.menu_img,
        link_type: menu.link_type,
        short_cut_keys: menu.short_cut_keys,
        order_no: menu.order_no,
        parent_id: menuData.parent_id,
        entity_id,
        company_id,
        created_by,
      },
    });
    console.log(`✅ Menu Item created: ${existingMenu.name}`);
  } else {
    existingMenu = await prisma.menuItems.update({
      where: { id: existingMenu.id },
      data: {
        menu_id: menu.menu_id,
        name: menu.name,
        target: menu.target,
        permission_id: permission_id,
        menu_img: menu.menu_img,
        link_type: menu.link_type,
        short_cut_keys: menu.short_cut_keys,
        order_no: menu.order_no,
        parent_id: menuData.parent_id,
        entity_id,
        company_id,
      },
    });
    console.log(`🔄 Menu Item updated: ${existingMenu.name}`);
  }

  if (children && children.length > 0) {
    for (const child of children) {
      const childMenu = { ...child, parent_id: existingMenu.id };
      await createMenu(company_id, childMenu, created_by);
    }
  }
}

export async function menuItemsSeeder(company_id: number) {
  try {
    console.log("🚀 Starting Menu Items Seeder...");

    let user = await prisma.user.findFirst({
      where: { company_id, role: 'super_admin' },
    });

    if (!user) {
      console.log(`❌ Menu Items seeder failed: No super_admin user for company_id ${company_id}`);
      return false;
    }

    const menus = JSON.parse(
      fs.readFileSync(path.join(__dirname, 'json/menuItems-seeds.json'), 'utf-8')
    );

    for (const menu of menus) {
      await createMenu(company_id, menu, user.id);
    }

    console.log("✅ Menu Items Seeder completed successfully.");
  } catch (error) {
    console.error(`❌ Menu Items Seeder Failed: ${error}`);
    throw error;
  } finally {
    await prisma.$disconnect();
  }
}
