import { PrismaClient, Prisma } from '@prisma/sql/client';
import fs from 'fs';
import path from 'path';

const prisma = new PrismaClient();

function formatDisplayName(displayName: string): string {
  return displayName
    .replace(/_/g, ' ') // Replace underscores with spaces
    .toLowerCase() // Convert to lower case
    .replace(/\b\w/g, (char) => char.toUpperCase()); // Capitalize first letter of each word
}

async function languageContent(
  languageContentData: { [key: string]: string },
  companyId: number,
  languageId: number
) {
  try {
    console.log(`🚀 Processing language content for company ${companyId}, language ${languageId}`);

    const displayNames = [
      ...(await prisma.masterEntityLineItem.findMany({
        where: { company_id: companyId, status_id: 1 },
        select: { display_name: true },
      })).map((item) => item.display_name),

      ...(await prisma.menus.findMany({
        where: { company_id: companyId, status_id: 1 },
        select: { name: true },
      })).map((item) => item.name),

      ...(await prisma.menuItems.findMany({
        where: { company_id: companyId, status_id: 1 },
        select: { name: true },
        orderBy: { order_no: 'asc' },
      })).map((item) => item.name),
    ];

    const keyValueMapping: { [key: string]: string } = {};

    for (const displayName of displayNames) {
      keyValueMapping[displayName] = formatDisplayName(displayName);
    }

    for (const [key, value] of Object.entries(languageContentData)) {
      keyValueMapping[key] = value;
    }

    for (const [key, value] of Object.entries(keyValueMapping)) {
      const existingContent = await prisma.languageContent.findFirst({
        where: { language_id: languageId, company_id: companyId, key_content: key },
      });

      if (!existingContent) {
        await prisma.languageContent.create({
          data: {
            language_id: languageId,
            key_content: key,
            values: value,
            company_id: companyId,
          },
        });
        console.log(`✅ Created languageContent: key ${key}, company_id ${companyId}`);
      } else {
        console.log(`🔄 Skipped existing languageContent: key ${key}, company_id ${companyId}`);
      }
    }
  } catch (error) {
    console.error(`❌ Error in languageContentSeeds: ${error}`);
    if (error instanceof Prisma.PrismaClientKnownRequestError && error.code === 'P2003' && error.meta) {
      console.error(`Foreign key constraint failed: ${error.meta.field_name}`);
    }
    throw error;
  }
}

export async function languageContentSeeds(company_id: number) {
  try {
    console.log("🚀 Starting Language Content Seeder...");

    const languages = [
      { path: 'json/language-content-english.json', languageId: 1 },
      { path: 'json/language-content-arabic.json', languageId: 2 },
    ];

    for (const { path: filePath, languageId } of languages) {
      const languageContentList: { [key: string]: string }[] = JSON.parse(
        fs.readFileSync(path.join(__dirname, filePath), 'utf-8')
      );

      for (const languageContentData of languageContentList) {
        await languageContent(languageContentData, company_id, languageId);
      }
    }

    console.log("✅ Language Content Seeds Completed.");
  } catch (error) {
    console.error(`❌ Language Content Seeder Failed: ${error}`);
    throw error;
  }
}
