import { PrismaClient } from '@prisma/sql/client';
import fs from 'fs';
import path from 'path';

const prisma = new PrismaClient();

const configurationsSeed = JSON.parse(
  fs.readFileSync(path.join(__dirname, 'json/configurations.json'), 'utf-8')
);

export async function configurationsSeeder(company_id: number) {
  try {
    console.log("🚀 Starting Configurations Seeder...");

    const user = await prisma.user.findFirst({
      where: { company_id, role: 'super_admin' },
    });

    if (!user) {
      console.log(`❌ Configurations Seeder failed: No super_admin user found for company_id ${company_id}`);
      return false;
    }

    const createdBy = user.id;
    const utcNow = new Date();

    for (const catType of configurationsSeed) {
      const categoryType = await prisma.categoryTypes.upsert({
        where: { company_id_name: { company_id, name: catType.cat_type_name } },
        update: {},
        create: {
          category_type_id: catType.category_type_id,
          name: catType.cat_type_name,
          description: catType.cat_type_description,
          company_id,
          created_by: createdBy,
          created_at: utcNow,
        },
      });

      if (catType.cats?.length) {
        for (const category of catType.cats) {
          const newCategory = await prisma.categories.upsert({
            where: { company_id_name: { company_id, name: category.cat_name } },
            update: {},
            create: {
              category_id: category.category_id,
              name: category.cat_name,
              description: category.cat_description,
              category_type_id: categoryType.category_type_id,
              company_id,
              created_by: createdBy,
              created_at: utcNow,
            },
          });

          if (category.cat_configs?.length) {
            await Promise.all(
              category.cat_configs.map(async (config) => {
                await prisma.configurations.upsert({
                  where: { company_id_config_key: { company_id, config_key: config.config_key } },
                  update: {},
                  create: {
                    order_no: config.order_no,
                    config_key: config.config_key,
                    config_value: config.config_value || 'null',
                    config_file_value: config.config_file_value ? JSON.stringify(config.config_file_value) : null,
                    config_value_type: config.config_value_type,
                    config_field_type: config.config_field_type,
                    category_id: newCategory.category_id,
                    company_id,
                    created_by: createdBy,
                    created_at: utcNow,
                  },
                });
              })
            );
          }
        }
      }
    }

    console.log("✅ Configurations Seeder completed successfully.");
  } catch (error) {
    console.error(`❌ Error in configurationsSeeder: ${error}`);
    throw error;
  } finally {
    await prisma.$disconnect();
  }
}
