import { PrismaClient } from '@prisma/pg/client';
const prisma = new PrismaClient();
import fs from 'fs';
const whatsappTemplateAssignmentSeeder = JSON.parse(fs.readFileSync(__dirname + '/json/whatsapp-template-assignment.json', 'utf-8'));

interface WhatsappTemplateData {
  whatsapp_template_process_slug: string;
  name?: string;
  recipient_type?: 'tag' | 'user_id' | 'custom';
  whatsapp_to?: string;
}

export async function whatsappTemplateAssignmentsSeeder(company_id: number) {
  try {
    let user = await prisma.user.findFirst({
      where: { company_id: company_id, role: 'super_admin' },
    });
    if (!user) {
      console.log(
        `Menu seeder failed: admin user not available for company id ${company_id}`,
      );
      return false;
    }
    const created_by = user.id;

    const datas: WhatsappTemplateData[] = whatsappTemplateAssignmentSeeder;

    for (const data of datas) {
        // Fetch the email_template_process_id using the provided slug and company_id
        const whatsappTemplateProcess = await prisma.whatsappTemplateProcess.findFirst({
            where: {
            slug: data.whatsapp_template_process_slug,
            company_id: company_id,
            },
        });

        if (!whatsappTemplateProcess) {
            console.error(
            `Whatsapp template process with slug "${data.whatsapp_template_process_slug}" not found for company_id ${company_id}.`,
            );
            continue;
        }

        // Initialize template IDs
        let template_id: number | null = null;

        // Fetch the user_template_id if template_slug is provided
        if (data.name) {
            const template = await prisma.whatsappTemplate.findFirst({
            where: {
                name: data.name,
                company_id: company_id,
            },
            });
            if (template) {
            template_id = template.id;
            } else {
            console.error(
                `User template with slug "${data.name}" not found for company_id ${company_id}.`,
            );
            continue;
            }
        }

        let whatsapp_to: string | null = `${data.whatsapp_to}` || null;
        if (data.recipient_type === 'tag') {
            whatsapp_to = null;
            if (data.whatsapp_to) {
                const template = await prisma.whatsappTemplateRecipientTag.findFirst({
                    where: {
                    slug: `${data.whatsapp_to}`,
                    company_id: company_id,
                    },
                });
                if (template) {
                    whatsapp_to = `${template.id}`;
                } else {
                    console.error(
                    `Email template recipient tag with slug "${data.whatsapp_to}" not found for company_id ${company_id}.`,
                    );
                    continue;
                }
            }
        }
        const whatsapp_template_process_id = whatsappTemplateProcess.id;

        const existingAssignment = await prisma.whatsappTemplateAssignment.findFirst(
            {
                where: {
                    whatsapp_template_process_id: whatsapp_template_process_id,
                    template_id: template_id || 0,
                    recipient_type: data.recipient_type,
                    whatsapp_to,
                    company_id: company_id,
                },
            },
        );

        if (!existingAssignment && template_id) {
            await prisma.whatsappTemplateAssignment.create({
            data: {
                whatsapp_template_process_id,
                template_id,
                recipient_type: data.recipient_type,
                whatsapp_to,
                company_id,
                created_by,
                created_at: new Date(),
            },
            });
            console.log(
            `Whatsapp Template Assignment with whatsapp_template_process_id "${whatsapp_template_process_id}" created for company_id ${company_id}.`,
            );
        } else {
            console.log(
            `Whatsapp Template Assignment with whatsapp_template_process_id "${whatsapp_template_process_id}" already exists for company_id ${company_id}.`,
            );
        }
    }
  }
  catch (error) {
    console.error('Error creating whatsapp template assignments:', error);
    throw error;
  } finally {
    // await prisma.$disconnect();
  }
}
