import { PrismaClient } from '@prisma/pg/client';
import { v4 as uuidv4 } from 'uuid';

const prisma = new PrismaClient();

function getCurrentUtcTime() {
  return new Date().toISOString();
}

const themeAttributes = [
  {
    attribute: 'accent_tone_color',
    value: 'rgb(97,15,16)',
    description: 'Header-bg-color,Button-hover-color,Header-color',
    status_id: 1,
  },
  {
    attribute: 'sidebar_heading_color',
    value: 'rgb(50, 45, 47)',
    description: 'Sidebar-heading-color',
    status_id: 1,
  },
  {
    attribute: 'accent_secondary_color',
    value: 'rgb(253,240,240)',
    description: 'Table-header-color,Pagination-button-bg-color',
    status_id: 1,
  },
  {
    attribute: 'sidebar_bg_color',
    value: 'rgb(34, 30, 32)',
    description: 'Sidebar-bg-color',
    status_id: 1,
  },
  {
    attribute: 'accent_secondary_tone_color',
    value: 'rgb(250, 225, 225)',
    description: 'Table-header-action-color',
    status_id: 1,
  },
  {
    attribute: 'accent_color',
    value: 'rgb(122, 18, 20)',
    description: 'Button-bg-color,Sidebar-hover-color,Form-base-component-color',
    status_id: 1,
  },
  {
    attribute: 'sidebar_text_color',
    value: 'rgb(255, 255, 255)',
    description: 'Sidebar-text-color,Header-text-color,Button-text-color',
    status_id: 1,
  },
];

export async function themeAttributeSeeder(company_id: number) {
  try {
    const utcNow = getCurrentUtcTime();

    const adminUser = await prisma.user.findFirst({
      where: { company_id, role: 'super_admin' },
    });

    if (!adminUser) {
      console.log(
        `Theme attribute seeder failed: super_admin not found for company_id ${company_id}`,
      );
      return false;
    }

    const createdBy = adminUser.id;

    const existingAttributes = await prisma.themeAttributes.findMany({
      where: { company_id, attribute: { in: themeAttributes.map(a => a.attribute) } },
    });

    const existingAttributeMap = new Map(
      existingAttributes.map(attr => [attr.attribute, attr]),
    );

    const promises: Promise<void>[] = [];

    for (const attr of themeAttributes) {
      const existingAttribute = existingAttributeMap.get(attr.attribute);

      if (existingAttribute) {
        promises.push(
          prisma.themeAttributes.update({
            where: { id: existingAttribute.id },
            data: {
              value: attr.value,
              description: attr.description,
              status_id: attr.status_id,
              updated_by: createdBy,
              updated_at: utcNow,
            },
          })
            .then(() =>
              console.log(`Theme attribute updated: ${attr.attribute}`),
            )
            .catch(err =>
              console.error(
                `Failed to update theme attribute ${attr.attribute}:`,
                err,
              ),
            ),
        );
      } else {
        promises.push(
          prisma.themeAttributes
            .create({
              data: {
                uuid: uuidv4(),
                attribute: attr.attribute,
                value: attr.value,
                description: attr.description,
                status_id: attr.status_id,
                company_id,
                created_by: createdBy,
                updated_by: createdBy,
                created_at: utcNow,
                updated_at: utcNow,
              },
            })
            .then(() =>
              console.log(`Theme attribute created: ${attr.attribute}`),
            )
            .catch(err =>
              console.error(
                `Failed to create theme attribute ${attr.attribute}:`,
                err,
              ),
            ),
        );
      }
    }

    await Promise.all(promises);

    console.log('Theme attribute seeding completed.');
    return true;
  } catch (error) {
    console.error('Theme attribute seeder failed:', error);
    throw error;
  } finally {
    // await prisma.$disconnect();
  }
}
