import { PrismaClient, ApproverType, Prisma } from '@prisma/pg/client';
import fs from 'fs';

const prisma = new PrismaClient();

function getCurrentUtcTime() {
  return new Date().toISOString();
}

export async function approvalWorkflow(company_id: number) {
  console.log("Seeding ApprovalWorkflow...");
  
  let user = await prisma.user.findFirst({
    where: { company_id: company_id, role: 'super_admin' }
  });
  if (!user) {
      console.log(`Department seeder failed: admin user not available for company id ${company_id}`);
      return false;
  }

  const createdBy = user.id;
  const utcNow = getCurrentUtcTime();

  const approvalWorkflows = [
     {
      slug: "user_timesheet_approval",
      name: "User Timesheets",
      description: "User Created TImesheet Approval workflow",
      query_information: `SELECT count(1) as exist FROM time_sheet t LEFT JOIN users u ON u.id = t.user_id where u.reporting_to = $prefill_data.users.id;`,
      url: 'timesheet/detail/',
      company_id,
      created_by: createdBy,
      updated_by: createdBy,
      created_at: utcNow,
      updated_at: utcNow,
    }
  ];

  // Insert data using upsert (to prevent duplicate entries)
  for (const workflow of approvalWorkflows) {
    await prisma.approvalWorkflow.upsert({
      where: { company_id_slug : {company_id, slug: workflow.slug} },
      update: workflow,
      create: workflow,
    });
    await approvalWorkflowAssignment(company_id, workflow.slug, createdBy, utcNow);
  }

  console.log("ApprovalWorkflow seeding completed!");
}


function fetchIdfromSlug(slug: string, list: {id: number, slug: string}[]): number {
  for(let each of list) {
    if(each.slug === slug) {
      return each.id;
    }
  }
  return 0;
}

function fetchIdfromName(name: string, list: {id: number, name: string}[]): number {
  for(let each of list) {
    if(each.name === name) {
      return each.id;
    }
  }
  return 0;
}

async function approvalWorkflowAssignment(company_id: number, slug:string, created_by: number, utcNow: string) {
    console.log("Seeding ApprovalWorkflowAssignment...");
    // Fetch existing approval workflows by slug
    const workFlow = await prisma.approvalWorkflow.findUnique({
      where: {company_id_slug : { company_id, slug }},
    });
  
    if (workFlow) {
        if(slug === 'user_timesheet_approval') {
          // Fetch existing email templates by slug
          const emailTemplates = await prisma.emailTemplate.findMany({
            where: {status_id : 1},
            select: {
              id: true,
              slug: true,
            }
          });

          // Fetch existing whatsapp templates by slug
          const whatsappTemplates = await prisma.whatsappTemplate.findMany({
            where: {status_id : 1},
            select: {
              id: true,
              name: true,
            }
          });
          
          const approvalAssignments = {
            approval_workflow_id: workFlow.id,
            approver_order_no: 1,
            approver_type: ApproverType.tag,
            approver: "reporting_manager",
            // approve_query_information: Prisma.JsonNull,
            reject_query_information: [ "update time_sheet set is_saved = false where id = $unique_id"],
            company_id,
            created_by: created_by,
            updated_by: created_by,
            created_at: utcNow,
            updated_at: utcNow,
            approve_mail_id: fetchIdfromSlug('timesheet_approval', emailTemplates),
            reject_mail_id: fetchIdfromSlug('timesheet-reject-request', emailTemplates),
            approve_whatsapp_id: fetchIdfromName('user_timesheet_approved', whatsappTemplates),
            reject_whatsapp_id: fetchIdfromName('user_timesheet_rejected', whatsappTemplates)
          }

          await prisma.approvalWorkflowAssignment.upsert({
            where: {
              company_id_approval_workflow_id_approver_order_no: {
                  company_id: approvalAssignments.company_id,
                  approval_workflow_id: approvalAssignments.approval_workflow_id,
                  approver_order_no: approvalAssignments.approver_order_no,
              }
            },
            update: approvalAssignments,
            create: approvalAssignments,
          });
        } else {
          const approvalAssignments = [
            {
                approval_workflow_id: workFlow.id,
                approver_order_no: 1,
                approver_type: ApproverType.tag,
                approver: "department_head",
                // approve_query_information: Prisma.JsonNull,
                // reject_query_information: Prisma.JsonNull,
                company_id,
                created_by: created_by,
                updated_by: created_by,
                created_at: utcNow,
                updated_at: utcNow,
            },
            {
                approval_workflow_id: workFlow.id,
                approver_order_no: 2,
                approver_type: ApproverType.user_id,
                approver: "1,2",
                // approve_query_information: Prisma.JsonNull,
                // reject_query_information: Prisma.JsonNull,
                company_id,
                created_by: created_by,
                updated_by: created_by,
                created_at: utcNow,
                updated_at: utcNow,
            },
            {
                approval_workflow_id: workFlow.id,
                approver_order_no: 3,
                approver_type: ApproverType.role_id,
                approver: "4",
                // approve_query_information: Prisma.JsonNull,
                // reject_query_information: Prisma.JsonNull,
                company_id,
                created_by: created_by,
                updated_by: created_by,
                created_at: utcNow,
                updated_at: utcNow,
            }
          ];
      
          // Insert data using upsert (to prevent duplicate entries)
          for (const assignment of approvalAssignments) {
          await prisma.approvalWorkflowAssignment.upsert({
                where: {
                  company_id_approval_workflow_id_approver_order_no: {
                      company_id: assignment.company_id,
                      approval_workflow_id: assignment.approval_workflow_id,
                      approver_order_no: assignment.approver_order_no,
                  }
                },
                update: assignment,
                create: assignment,
            });
          }
      }
      console.log("ApprovalWorkflowAssignment seeding completed!");
    }
  
  }

/*
#INSERT
INSERT INTO approval_process_jobs 
(approval_workflow_slug, name, primary_table, unique_id, user_id, prefill_data, company_id, status_id) 
VALUES 
(
  'order_greaterthan_10000', 
  'Order > 10000 Approval', 
  'orders', 
  1, 
  1, 
  '{}'::jsonb, -- Empty JSON object instead of NULL
  1,
  1 -- Default status
)
ON CONFLICT (company_id, id) DO NOTHING;

#UPDATE
UPDATE approval_process_jobs 
SET 
  name = 'Order > 10000 Approval',
  primary_table = 'orders',
  unique_id = 1,
  user_id = 1,
  prefill_data = '{}'::jsonb, -- Empty JSON object instead of NULL
  status_id = 1
WHERE 
  approval_workflow_slug = 'order_greaterthan_10000'
  AND company_id = 1;
*/