import { PrismaClient } from '@prisma/pg/client';
import { v4 as uuidv4 } from 'uuid';

const prisma = new PrismaClient();

function getCurrentUtcTime() {
  return new Date().toISOString();
}

export async function appVersionsSeeder(company_id: number) {
  try {
    const utcNow = getCurrentUtcTime();

    // Fetch the super_admin user
    const user = await prisma.user.findFirst({
      where: { company_id, role: 'super_admin' },
    });

    if (!user) {
      console.log(
        `App Versions seeder failed: admin user not available for company id ${company_id}`,
      );
      return false;
    }

    // Sample app versions data
    const appVersions = [
      {
        type: 'lcp',
        version_type: 'major',
        version: '1.0.0',
        description: `<p>1. Released version module under admin-menu -> about LCP.</p>  
                      <p>2. Added "version_grid" at master entity level.</p>  
                      <p>3. Seeder added for this version.</p>`,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Mukesh M',
        created_at: new Date('2025-03-14 11:33:44.012+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'minor',
        version: '1.1.0',
        description: 'Design changes implemented.',
        is_frontend_changed: true,
        is_backend_changed: false,
        is_database_changed: false,
        developed_by: 'Mukesh M',
        created_at: new Date('2025-03-14 11:33:44.012+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'patch',
        version: '1.1.1',
        description: 'Seeder updated for "app_version_grid".',
        is_frontend_changed: false,
        is_backend_changed: true,
        is_database_changed: false,
        developed_by: 'Mukesh M',
        created_at: new Date('2025-03-14 11:33:44.012+05:30'),
      },
      
      {
        type: 'lcp',
        version_type: 'major',
        version: '2.0.0',
        description: `<p>1. Azure Configuration changes in Configuration screen. </p> 
                      <p>2. Powerbi Report embed changes in Master Entity and Dashboard Modules. </p> 
                      <p>3. Auto Logout Functionality based on Token Expiration and Idle Timeout and added configuration changes in Configuration screen.</p>
                      <p>4. Updated the action 'Export' as 'Export As Excel' in Master Entity and User/Role Permission screens.</p>
                      <p>5. Added Developed By and Created At column in the App Version screen and updated the db schema.</p>
                      <p>6. Include Preview option in Rich Text Editor Formly Component for Email Template and inlcuded the preview parameter in the form information.</p>
                      <p>7. Updated the delete icon in the Menu Mapping screen.</p>
                      <p>8. Included Sample JSON Template as popup for the query information, form information, report information, add/edit information and preset information in the Master entity screen and in Policy Creation screen. </p>
                      <p>9. Updated the get_grid_item procedure for the data table sorting issue.</p>
                      <p>10. Language Content and Seeder data updated.</p>
                      <p>11. Language Content Translation issue.</p>`,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Logesh T',
        created_at: new Date('2025-03-20 00:00:00.000+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'patch',
        version: '2.0.1',
        description: `<p>1. Lazy Loading for Dynamic Routes.  </p>
                      <p>2. Included APP and LCP Latest Version Section in the settings Nav Bar.</p>`,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Logesh T',
        created_at: new Date('2025-03-21 00:00:00.000+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'major',
        version: '3.0.0',
        description: `<p>1. Released cron setting module under admin-menu -> cron settings.  </p>
                      <p>2. Its dynamic cron setup module.  </p>
                      <p>3. Now no need server setup.</p>`,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'karthik N,Mukesh M',
        created_at: new Date('2025-04-11 11:33:44.012+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'major',
        version: '4.0.0',
        description: `<p>1. Released AI Playgorund module under tools menu -> schema chunks,</p>
                      <p>2. Added Schema Chunks Generate and Edit Module,</p>
                      <p>3. Added OpenAI API Key Configurations,</p>
                      <p>4. The purpose of this module is to enable users to generate sql based on the natural language,  </p>
                      <p>5. During initial setup, don't forgot to click on sync table schema and generate vector for all chunks buttons from schema chunks module.</p>`,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Mithilesh Krishna S',
        created_at: new Date('2025-04-17 11:33:44.012+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'major',
        version: '5.0.0',
        description: `<p>1. Approval Workflow and Tags modules released under admin-menu -> approval workflow and approver tags.
                      <p>2. Approver Workflow assignment, Approval Request History and Delegation modules released under tools-menu -> approval workflow assignment, approval requests and delegations.
                      <p>3. Updated the master entity screen with the draft otpion and updated the add / edit JSON with the approval workflow.
                      <p>4. The purpose of these modules is to create a approval process for a transaction which can have multiple approver levels and inlcuded the delegation functionality which is used to assign an alternative user if certain approver is not available for certain duration.,  
                      <p>5. In order to use approval process, draft option needs to be enabled for the respective screens and include the list of approval process in the add / update query information using its slug name in the form module section in master entity.
                      <p>6. Approve and reject query options has been included in the approval workflow assignment screen to perform certain action in each level.
                      <p>7. Updated the Grid and Form Builder Module to show approval process details.`,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Logesh T',
        created_at: new Date('2025-04-21 11:33:44.012+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'patch',
        version: '5.0.1',
        description: `<p>1. Placeholder ($session_user_id) replacement logic has been included in the get_grid_item procedure which is applicable to search_all, search_any, having_all_conditions, having_any_conditions and join_conditions in includes for grid module.</p>
                      <p>2. Replaced $user_id placeholder with $session_user_id in form seeders.</p>
                      <p>3. Included $unique_id and $session_user_id placeholder replacement logic for policies.</p>
                      <p>4. Created Team Settings under Tools Menu where these placeholder replacement logic has been implemented as an example.</p>
                      `,
        is_frontend_changed: false,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Logesh T',
        created_at: new Date('2025-04-23 11:33:44.012+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'minor',
        version: '5.1.0',
        description: 'Authentication Layout changes',
        is_frontend_changed: true,
        is_backend_changed: false,
        is_database_changed: false,
        developed_by: 'Rajkumar Shanmugavel',
        created_at: new Date('2025-04-23 17:01:13.745124+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'major',
        version: '6.0.0',
        description: `<p>1. Released Query Builder module under tools menu -> Query Builder,</p>
                      <p>2. We can individually select, update, delete or perform other sql operations,</p>
                      <p>3. Added Textare to store the results,</p>
                      <p>4. Listed all tables and Grid Table to display the result,</p>
                      <p>5. The purpose of this module is to enable users to perform sql opertions on DB from frontend.</p>`,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Mithilesh Krishna S',
        created_at: new Date('2025-04-25 11:33:44.012+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'minor',
        version: '6.1.0',
        description: `
        <p>1. Added a new tables as child_processes and child_process_logs.</p>
        <p>2. Backgorund process and Backgorund Process Logs modules have been added under Admin Module  -> Backgorund Settings -> Backgorund Processes and Backgorund Process Logs.</p>
        <p>3. The purpose of these modules is to enable the users to create a Backgorund process which can be used to perform certain tasks in the background.</p>
        `,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Logesh T',
        created_at: new Date('2025-05-08 17:01:13.745124+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'minor',
        version: '6.2.0',
        description: `
        <p>1. Added a new columns to store approval and reject mail template id in approval workflow,</p>
        <p>2. Modified initiate_approval_process trigger and execute_approval_process trigger and added new function to fetch email templates,</p>
        <p>3. Modified UI on Approval Assignment to accomodate approval and rejection workflow  </p>
        `,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Mithilesh Krishna S',
        created_at: new Date('2025-05-13 11:00:00.745124+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'minor',
        version: '6.3.0',
        description: `
        <p>1. Added a new columns to store details in approval_process_jobs and approval_process_jobs_workflow workflow,</p>
        <p>2. Approval Workflow Add Approve, Reject buttons in status tab,</p>
        <p>3. Avoid opening page link externally </p>
        `,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Mithilesh Krishna S',
        created_at: new Date('2025-05-19 11:00:00.745124+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'minor',
        version: '6.4.0',
        description: `
        <p>1. Added a new columns to store expression rules & messages in import template line items,</p>
        `,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Rajkumar S',
        created_at: new Date('2025-05-25 12:30:00.745124+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'patch',
        version: '6.4.1',
        description: `
        <p>1. Added <render-html> in page builder module,</p>
        <p>2. Formatted JSON, </p>
        `,
        is_frontend_changed: true,
        is_backend_changed: false,
        is_database_changed: false,
        developed_by: 'Mithilesh Krishna S',
        created_at: new Date('2025-05-28 11:00:00.745124+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'minor',
        version: '6.5.0',
        description: `
       <p>1. Cron job - Audit Logs Truncate (3 days once) - done,</p>
<p>2. Cron job - Request Logs Truncate (3 days once) - done,</p>
<p>3. LCP login video - Seeder level - done,</p>
<p>4. Child process setting - Background Process - all - done,</p>
<p>5. Background process list - created_at column added - done,</p>
<p>6. Each process - overall checkbox - All levels role map - done,</p>
<p>7. Scheduled Job Logs page - Implemented - done,</p>
<p>8. Table scheduled_job_logs added with columns - done:</p>
<p>&nbsp;&nbsp;- id,</p>
<p>&nbsp;&nbsp;- uuid,</p>
<p>&nbsp;&nbsp;- scheduled_job_id (foreign key),</p>
<p>&nbsp;&nbsp;- executed_statement,</p>
<p>&nbsp;&nbsp;- status,</p>
<p>&nbsp;&nbsp;- response,</p>
<p>&nbsp;&nbsp;- created_at,</p>
<p>9. Add description field in schedule_job - done,</p>
<p>10. Name changes: truncate-request*** instead of process-**** - done,</p>
<p>11. LCP filter case COALESCE problem - done,</p>
<p>12. Return query execution even on error - done,</p>
<p>13. Set Back button when detail grid comes - done,</p>
<p>14. Implement unique_id in get grid items in both frontend and backend - done,</p>
<p>15. Add description field in schedule_job frontend (add/edit forms) and backend (Seeder) - done,</p>

        `,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Rajkumar Shanmugavel',
        created_at: new Date('2025-06-09 08:00:00.745124+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'patch',
        version: '6.5.1',
        description: `
        <p>1. Added LIMIT in get_multiple_records procedure,</p>
        <p>2. Added OFFSET in get_multiple_records procedure, </p>
        `,
        is_frontend_changed: false,
        is_backend_changed: false,
        is_database_changed: true,
        developed_by: 'Mithilesh Krishna S',
        created_at: new Date('2025-06-11 11:00:00.745124+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'patch',
        version: '6.5.2',
        description: `
    <p><strong>13/06/2025</strong></p>
    <p>1. Updated info popup content for default Grid Builder in Master Entity Form Page.</p>
    <p>2. Removed default sort columns in Master Entity List Page.</p>
    <p>3. Implemented default <code>sort_columns</code> in all grid JSON (backend seeder).</p>
    <p>4. Modified get grid items procedure to prioritize frontend sort columns and fallback to DB <code>query_information</code>.</p>
    <p>5. Seeder: Background process pages changed to nested grid layout (e.g., Cron Settings page shows job logs in same page).</p>
    <p>6. Seeder: Menu name changed from "Cron Settings" to "Cron Jobs", and from "Background Process" to "Background Jobs". Grouped under "Background Process Settings".</p>
    <p>7. Fixed Policy User Mapping screen issue - now allows saving even when all users are deselected.</p>
    <p>8. Payload encryption and decryption implemented at both frontend and backend levels.</p>

    <p><strong>17/06/2025</strong></p>
    <p>9. Added encryption tokens in environment files (both frontend and backend).</p>
   
  `,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Rajkumar Shanmugavel',
        created_at: new Date('2025-06-13 11:00:00.000+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'patch',
        version: '6.5.3',
        description: `
    <p><strong>17/06/2025</strong></p>
   
    <p>1. Implemented parent-child datatable handling at LCP level.</p>
  `,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Rajkumar Shanmugavel',
        created_at: new Date('2025-06-17 11:00:00.000+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'major',
        version: '7.0.0',
        description: `<p>1. whatsapp cloud API for messages implementation</p>
        <p>2. whatsapp business account creation</p>
        <p>3. Seed for whatsapp process, tags, templates and assignments</p>
        <p>4. Custom whatsapp assignment UI</p>`,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Mithilesh Krishna S',
        created_at: new Date('2025-06-23 11:00:00.000+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'patch',
        version: '7.0.1',
        description: `
    <p><strong>23/06/2025</strong></p>
    <p>1. Implemented an additional plus button to show the modal for the add form within each form group.</p>
    <p>2. Common link_type and link_value setup in the master_entity page for entity_grid to open a modal or redirect to a URL. [done]</p>
    <p>3. Implemented CTE in query_information for both frontend and backend (seeder grid data and list data). [done]</p>
    <p>4. Updated the CTE logic in the UI information popup. [done]</p>
    <p>5. User session logout now applies to all open tabs and restricts operations post logout.</p>
  `,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Rajkumar Shanmugavel',
        created_at: new Date('2025-06-23 12:00:00.000+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'patch',
        version: '7.0.2',
        description: `
    <p><strong>24/06/2025</strong></p>
    <p>1. Store created_at, updated_at, and deleted_at in UTC format at DB level.</p>
    <p>2. Implement TimeZone dropdown for display.</p>
    <p>3. Implement DateTime format dropdown for display.</p>
  `,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Rajkumar Shanmugavel',
        created_at: new Date('2025-06-24 10:00:00.000+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'patch',
        version: '7.0.3',
        description: `
          <p><strong>03/07/2025</strong></p>
          <p>1. "+" icon popup form now closes without refreshing the page. (Adding value to dropdown was pending earlier, now completed)</p>
          <p>2. Fixed "+" icon popup in Edit User screen to correctly insert value into the database.</p>
          <p>3. Increased popup close button size and made it configurable.</p>
          <p>4. Approval request "Approve" and "Reject" buttons are now responsive in size.</p>
          <p>5. Dashboard widget popup (users, roles, etc.) shows detailed table screen on number click.</p>
          <p>6. Default app version seeder implemented with model seeding support.</p>
          <p>7. Fetch <code>print_query</code> from DB and apply in procedure with default true.</p>
          <p>8. Set popup modal close button at the top-right corner.</p>
          <p>9. Fixed dashboard modal popup to hide grid content when not needed.</p>
          <p>10. (Pending) Configurable popup options (size, overlay, loader, etc.).</p>
        `,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Rajkumar Shanmugavel',
        created_at: new Date('2025-07-03 10:00:00.000+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'patch',
        version: '7.0.4',
        description: `
          <p><strong>23/07/2025</strong></p>
          <p>1. Enabled grid column click to open detailed data view (column click grid details).</p>
          <p>2. Introduced user-based configurations to allow personalized behavior/settings per user.</p>
        `,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: false,
        developed_by: 'Rajkumar Shanmugavel',
        created_at: new Date('2025-07-23 10:00:00.000+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'minor',
        version: '7.1.1',
        description: `
          <p><strong>02/09/2025</strong></p>
          <p>1. Added Theme and Theme Attribute settings in admin menu.</p>
          <p>2. Implemented Theme switching functionality and add css for customization.</p>
        `,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Kharthie Ramakrishnan',
        created_at: new Date('2025-09-02 10:00:00.000+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'patch',
        version: '7.1.2',
        description: `
          <p><strong>11/09/2025</strong></p>
          <p>1. Automated database procedure execution with support for:</p>
          <ul>
            <li>Single execution with stop-on-failure</li>
            <li>Transactional execution with rollback-on-failure</li>
            <li>Dry-run mode to preview files before execution</li>
            <li>App-specific procedure execution commands</li>
            <li>Run only / exclude selected SQL files from command line</li>
          </ul>
          
        `,
        is_frontend_changed: false,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Rajkumar Shanmugavel',
        created_at: new Date('2025-09-11 10:00:00.000+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'minor',
        version: '7.2.0',
        description: `
          <p><strong>11/09/2025</strong></p>
          <p>1. The Barcode module has been added to the Admin menu.</p>
          <p>2. You can now create barcode templates using this module.</p>
          <p>3. The print functionality supports generating barcodes and printing RFID data.</p>
        `,
        is_frontend_changed: true,
        is_backend_changed: true,
        is_database_changed: true,
        developed_by: 'Logesh T',
        created_at: new Date('2025-09-11 10:00:00.000+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'patch',
        version: '7.2.1',
        description: `
          <p><strong>12/09/2025</strong></p>
          <p>1. Implemented a date field and dropdown in dashboard component.</p>
          <p>2. Implemented a date range filter for the dashboard data.</p>
        `,
        is_frontend_changed: true,
        is_backend_changed: false, 
        is_database_changed: false,
        developed_by: 'Kharthie Ramakrishnan',
        created_at: new Date('2025-09-12 10:00:00.000+05:30'),
      },
      {
        type: 'lcp',
        version_type: 'patch',
        version: '7.2.2',
        description: `
          <p><strong>17/09/2025</strong></p>
          <p>1. Implemented a dynamic translate in label and placeholder in form_builder_module (forms).</p>
          <p>2. Passing the key inside the label and placeholder in form_information json.</p>
          <p>1. Implemented a dynamic translate using hbp_translate in static_page_builder_module (Details Page).</p>
          <p>2. Passing the key inside the static_page_builder_module HTML.</p>
        `,
        is_frontend_changed: true,
        is_backend_changed: true, 
        is_database_changed: false,
        developed_by: 'Kharthie Ramakrishnan',
        created_at: new Date('2025-09-17 10:00:00.000+05:30'),
      }
    ];

    for (const appVersion of appVersions) {
      await createOrUpdateAppVersion(company_id, appVersion, user.id, utcNow);
    }

    console.log('App Versions seeder executed successfully.');
  } catch (error) {
    console.error(`Error in appVersionsSeeder: ${error}`);
    throw error;
  }
}

async function createOrUpdateAppVersion(
  company_id: number,
  appVersion: any,
  created_by: number,
  utcNow: string,
) {
  // Check if the app version already exists by type and version
  let existingVersion = await prisma.appversions.findFirst({
    where: {
      company_id,
      type: appVersion.type,
      version_type: appVersion.type,
      version: appVersion.version,
    },
  });

  if (!existingVersion) {
    // If no existing version, create a new one
    const newVersion = await prisma.appversions.create({
      data: {
        company_id,
        type: appVersion.type,
        version_type: appVersion.version_type,
        version: appVersion.version,
        description: appVersion.description,
        is_frontend_changed: appVersion.is_frontend_changed,
        is_backend_changed: appVersion.is_backend_changed,
        is_database_changed: appVersion.is_database_changed,
        developed_by: appVersion.developed_by,
        created_at: appVersion.created_at,
        created_by,
      },
    });

    console.log(`LCP and App version created: ${newVersion.version}`);
  } else {
    // If exists, update it
    await prisma.appversions.update({
      where: { id: existingVersion.id },
      data: {
        description: appVersion.description,
        is_frontend_changed: appVersion.is_frontend_changed,
        is_backend_changed: appVersion.is_backend_changed,
        is_database_changed: appVersion.is_database_changed,
        developed_by: appVersion.developed_by,
        created_at: appVersion.created_at,
        updated_by: created_by,
        updated_at: utcNow,
      },
    });

    console.log(`LCP and App version updated: ${existingVersion.version}`);
  }
}
