import { PrismaClient } from '@prisma/pg/client';
import { v4 as uuidv4 } from 'uuid';

const prisma = new PrismaClient();

function getCurrentUtcTime() {
  return new Date().toISOString();
}

export async function appVersionsSeeder(company_id: number) {
  try {
    const utcNow = getCurrentUtcTime();

    // Fetch the super_admin user
    const user = await prisma.user.findFirst({
      where: { company_id, role: 'super_admin' },
    });

    if (!user) {
      console.log(
        `App Versions seeder failed: admin user not available for company id ${company_id}`,
      );
      return false;
    }

    // Sample app versions data
    const appVersions = [{
      type: 'app',
      version_type: 'major',
      version: '1.0.0',
      description: `<p>1. Released version module under admin-menu -> about LCP and app.</p>  
                    <p>2. Added "app_version_grid" at master entity level.</p>  
                    <p>3. Seeder added for this version.</p>`,
      is_frontend_changed: true,
      is_backend_changed: true,
      is_database_changed: true,
      developed_by: 'Mukesh M',
      created_at: new Date('2025-03-14 11:33:44.012+05:30'),
    }];

    for (const appVersion of appVersions) {
      await createOrUpdateAppVersion(company_id, appVersion, user.id, utcNow);
    }

    console.log('App Versions seeder executed successfully.');
  } catch (error) {
    console.error(`Error in appVersionsSeeder: ${error}`);
    throw error;
  }
}

async function createOrUpdateAppVersion(
  company_id: number,
  appVersion: any,
  created_by: number,
  utcNow: string,
) {
  // Check if the app version already exists by type and version
  let existingVersion = await prisma.appversions.findFirst({
    where: {
      company_id,
      type: appVersion.type,
      version_type: appVersion.type,
      version: appVersion.version,
    },
  });

  if (!existingVersion) {
    // If no existing version, create a new one
    const newVersion = await prisma.appversions.create({
      data: {
        company_id,
        type: appVersion.type,
        version_type: appVersion.version_type,
        version: appVersion.version,
        description: appVersion.description,
        is_frontend_changed: appVersion.is_frontend_changed,
        is_backend_changed: appVersion.is_backend_changed,
        is_database_changed: appVersion.is_database_changed,
        developed_by: appVersion.developed_by,
        created_at: appVersion.created_at,
        created_by,
      },
    });

    console.log(`LCP and App version created: ${newVersion.version}`);
  } else {
    // If exists, update it
    await prisma.appversions.update({
      where: { id: existingVersion.id },
      data: {
        description: appVersion.description,
        is_frontend_changed: appVersion.is_frontend_changed,
        is_backend_changed: appVersion.is_backend_changed,
        is_database_changed: appVersion.is_database_changed,
        developed_by: appVersion.developed_by,
        created_at: appVersion.created_at,
        updated_by: created_by,
        updated_at: utcNow,
      },
    });

    console.log(`LCP and App version updated: ${existingVersion.version}`);
  }
}
