import { PrismaClient } from '@prisma/pg/client';
import fs from 'fs';

const prisma = new PrismaClient();

function getCurrentUtcTime() {
  return new Date().toISOString();
}

// Load the push and pull data transfer seeds from JSON files
const pushDataTransferSeed = JSON.parse(
  fs.readFileSync(__dirname + '/json/data-transfer-queries-push.json', 'utf-8'),
);

const pullDataTransferSeed = JSON.parse(
  fs.readFileSync(__dirname + '/json/data-transfer-queries-pull.json', 'utf-8'),
);

export async function dataTransferQueries(company_id: number) {
  try {
    // Fetch a super admin user for the given company
    const user = await prisma.user.findFirst({
      where: { company_id: company_id, role: 'super_admin' },
    });

    // Check if the super admin user exists
    if (!user) {
      console.log(
        `Role seeder failed: admin user not available for company id ${company_id}`,
      );
      return false; // Return early if the user does not exist
    }

    const createdBy = user.id;
    const utcNow = getCurrentUtcTime();

    // Seed push data transfer queries
    for (const item of pushDataTransferSeed) {
      await prisma.dataTransferQueries.upsert({
        where: {
          id: item.id, // Assuming id is unique for each item
        },
        update: {},
        create: {
          ...item,
          company_id: company_id,
          created_by: createdBy,
          created_at: utcNow,
        },
      });
    }

    // Optionally, you can implement the same for pull data transfer queries
    for (const item of pullDataTransferSeed) {
      await prisma.dataTransferQueries.upsert({
        where: {
          id: item.id, // Assuming id is unique for each item
        },
        update: {},
        create: {
          ...item,
          company_id: company_id,
          created_by: createdBy,
          created_at: utcNow,
        },
      });
    }

    console.log('Data transfer queries seeded successfully.');
  } catch (error) {
    console.error(`Error in configurationsSeeder: ${error}`);
    throw error; // Rethrow the error for handling in the calling context
  } finally {
    // await prisma.$disconnect(); // Ensure Prisma Client disconnects
  }
}
