import { PrismaClient } from '@prisma/pg/client'
import fs from 'fs';
const prisma = new PrismaClient();

function getCurrentUtcTime() {
  return new Date().toISOString();
}

async function getMasterEntityIdBySlug(
  permission_slug: string,
  company_id: number,
) {
  const masterEntity = await prisma.masterEntity.findFirst({
    where: {
      company_id,
      entity_name: permission_slug,
    },
  });
  return masterEntity ? masterEntity.id : null;
}


async function getMasterPermissionIdBySlug(
  permission_slug: string,
  company_id: number,
) {
  const masterEntity = await prisma.permissions.findFirst({
    where: {
      company_id,
      slug: permission_slug,
    },
  });
  return masterEntity ? masterEntity.id : null;
}

async function createMenu(company_id: number, menu: any, created_by:number) {
  const { children, entity_name, permission_slug, ...menuData } = menu;
  // Check if the menu already exists
  let existingMenu = await prisma.menuItems.findFirst({
    where: { company_id, name: menu.name, parent_id: menuData.parent_id },
  });

  let entity_id: number | null = null;
  if (entity_name) {
    entity_id = await getMasterEntityIdBySlug(entity_name, company_id);
  }

  let permission_id: number | null = null;
  if (permission_slug) {
    permission_id = await getMasterPermissionIdBySlug(
      permission_slug,
      company_id,
    );
  }

  // If it does not exist, create it
  if (!existingMenu) {
    const data = {
      menu_id:menu.menu_id,
      name: menu.name,
      target: menu.target,
      permission_id: permission_id,
      menu_img: menu.menu_img,
      link_type: menu.link_type,
      short_cut_keys: menu.short_cut_keys,
      order_no: menu.order_no,
      parent_id: menuData.parent_id,
      entity_id,
      company_id,
      created_by
    };
    existingMenu = await prisma.menuItems.create({
      data
    });
    console.log(`Mobile Menu Items created successfully: ${existingMenu.name}`);
  }
  else{
    const data = {
      menu_id:menu.menu_id,
      name: menu.name,
      target: menu.target,
      permission_id: permission_id,
      menu_img: menu.menu_img,
      link_type: menu.link_type,
      short_cut_keys: menu.short_cut_keys,
      order_no: menu.order_no,
      parent_id: menuData.parent_id,
      entity_id,
      company_id
    };

    menu = await prisma.menuItems.update({
      where: { company_id_id: { company_id, id: existingMenu.id } },
      data,
    });
    console.log(`Mobile Menu Items updated successfully: ${menu.name}`);
  }

  // Recursively create children
  if (children && children.length > 0) {
    for (const child of children) {
      const childMenu = { ...child, parent_id: existingMenu.id };
      await createMenu(company_id, childMenu, created_by);
    }
  }
}

export async function mobileMenuItemsSeeder(company_id: number) {
  try {
    const utcNow = getCurrentUtcTime();
    let user = await prisma.user.findFirst({
      where: { company_id, role: 'super_admin' },
    });
    if (!user) {
      console.log(
        `Mobile Menu Items seeder failed: admin user not available for company id ${company_id}`,
      );
      return false;
    }
    const menus = JSON.parse(
      fs.readFileSync(__dirname + '/json/mobileMenuItems-seeds.json', 'utf-8'),
    );
    for (const menu of menus) {
      await createMenu(company_id, menu, user.id);
    }
  } catch (error) {
    throw error;
  } finally {
    // await prisma.$disconnect();
  }
}
